# Documentation Writing Guide

This guide defines standards for writing and maintaining technical documentation in `docs/`.

## Purpose

Documentation should:
- **Explain concepts and architecture**, not list code details
- **Be verifiable** against the codebase
- **Stay accurate** as code evolves
- **Help developers understand** how the system works

## File Naming Convention

```
{sequence}_{category}_{topic}.md

Examples:
01_1_ARCH_overview.md
01_4_ARCH_package-system.md         # Main document
01_4_1_ARCH_build-phases.md         # Subsection (optional)
02_3_PLUGIN_mvc-pattern.md
06_2_INSTALLER_extraction.md
99_1_APPENDIX_doc-writing-guide.md
```

**Categories:**
- `ARCH` - Architecture and core concepts
- `PLUGIN` - Plugin structure and WordPress integration
- `UI` - User interface (admin pages, templates)
- `PACKAGE` - Package creation system
- `STORAGE` - Storage providers and upload system
- `INSTALLER` - Standalone installer architecture
- `SECURITY` - Security patterns and best practices
- `DEV` - Development practices and utilities
- `OPS` - Operations (testing, building, deployment)
- `APPENDIX` - Meta-documentation (guides, tools, practices)

**Sequence numbers:**
- Major category: `01_*`, `02_*`, etc.
- Files within category: `01_1`, `01_2`, `01_3`, etc.
- Appendix: `99_*` range

**Subsections (optional third level):**

When a topic becomes too complex or the document grows too large, consider splitting it into subsections:

```
01_4_ARCH_package-system.md       ← Main document (overview)
01_4_1_ARCH_build-phases.md       ← Detailed subsection
01_4_2_ARCH_archive-creation.md   ← Detailed subsection
01_4_3_ARCH_storage-upload.md     ← Detailed subsection
```

**When to use subsections:**
- Main document exceeds ~300 lines or covers multiple distinct workflows
- Topic has clear, independent sub-topics (e.g., different archive formats, build phases)
- Readers benefit from focused, detailed documents vs one large reference

**Structure:**
- **Main document** (`01_4`): High-level overview, common concepts, links to subsections
- **Subsections** (`01_4_1`, `01_4_2`): Deep dive into specific workflows, edge cases, detailed examples

**Key principle:** Subsections maintain the same category and relate to the parent topic. Use sparingly - most topics fit in a single document.

## Writing Style

### Tone

- **Clear and direct** - Avoid ambiguity, verbosity, and repetition
- **Concise** - Respect reader's time
- **Technical but accessible** - Assume PHP/WordPress knowledge, explain domain concepts
- **Factual** - State what is, not opinions about quality

### Formatting

**Emphasis:**
- **Bold** for important terms, component names
- `Code formatting` for class names, method names, file paths, namespaces
- _Italics_ sparingly, for subtle emphasis

**Lists:**
- Use bullet points for unordered information
- Use numbered lists for sequential steps
- Keep list items parallel in structure

**Code blocks:**
- Always specify language: ` ```php `, ` ```bash `
- Include context (what the code does)
- Keep examples short and focused

**Tables:**
Use for structured data (configurations, comparisons):

```markdown
| Class | Namespace | Purpose |
|-------|-----------|---------|
| `Bootstrap` | `Duplicator\Core` | Plugin initialization |
```

## Writing Verifiable Documentation

### ✅ Good: Concept-Focused Claims

Write about **what** the system does and **how** it works conceptually:

```markdown
- "Packages contain a database export and file archive"
- "Storage uploads are tracked via UploadInfo entities"
- "The installer runs independently of WordPress"
- "AbstractStorageEntity provides the base for all storage providers"
- "TplMng handles template rendering using MVC pattern"
```

**Why:** These claims describe architecture, behavior, and patterns that remain true even if class names or file paths change.

### ❌ Bad: Implementation-Specific Details

Avoid brittle references that break easily:

```markdown
- "The PackageProcessor class is located at src/Package/PackageProcessor.php"
- "Line 245 of BuildProgress.php handles status updates"
- "The method is called createArchive(string $path, array $files)"
- "Import the class using: use Duplicator\Package\Build;"
```

**Why:** File paths, line numbers, and exact signatures change frequently and make documentation fragile.

### ⚖️ When to Include Implementation Details

**Include specific references when:**
- Referring to a **core architectural component** (e.g., "AbstractStorageEntity base class")
- Documenting a **public API or pattern** others must use
- Showing a **code example** that illustrates the concept

**Format:**
```markdown
**Component:** AbstractStorageEntity
**Namespace:** `Duplicator\Models`
**Purpose:** Base class for all storage provider implementations

**Usage Example:**
```php
class CustomStorage extends AbstractStorageEntity
{
    public function upload($package)
    {
        // Implementation details
    }
}
```

**Key principle:** Reference implementation details when they help understanding, but focus on the concept and purpose.

### Verifiable vs Non-Verifiable Claims

Documentation should be written so that its accuracy can be **automatically verified** against the codebase.

**✅ Good verifiable claims:**
- State relationships between components
- Describe data flow sequences
- List system components and their purposes
- Explain behavioral rules and constraints
- Reference core architectural patterns

**Example of verifiable documentation:**
```markdown
1. Package build begins with pre-process validation
2. Scanner identifies files to include in archive
3. DatabasePkg exports database to SQL file
4. Archive is created using DupArchive or ZIP format
5. Package is uploaded to configured storage locations
```

Why this is verifiable: Can check build phases in code, find DatabasePkg class, verify archive formats, confirm storage upload logic.

**❌ Non-verifiable (avoid):**
```markdown
The package creation is fast and efficient. It uses best practices
and follows industry standards. The code is well-organized.
```

Why this fails: "Fast", "efficient", "best practices", "well-organized" are subjective opinions with no concrete claims to verify.

## Common Mistakes

### ❌ Mistake #1: Too Much Code

**RED FLAG: Code block exceeds 10 lines?** You're documenting implementation, not concepts.

**Examples of TOO MUCH**:
- Copying entire function bodies (20+ lines)
- Complete method signatures with all parameters
- Full class implementations

**Rule**: 3-5 lines max. Show the pattern, not the implementation.

## What to Document

### ✅ Do Document

- **Architecture and patterns** - How the system is structured
- **Core components** - Package system, storage, installer, entities
- **Data flows** - How data moves through the system
- **Public APIs** - Interfaces other developers use
- **Business logic** - Why certain decisions are made
- **Integration points** - WordPress hooks, storage APIs, etc.
- **Configuration** - Important settings and their impact
- **Non-obvious behavior** - Things that aren't self-evident
- **Behavioral rules** - What the system enforces

### ❌ Don't Document

- **Implementation details** - Specific method internals (use code comments)
- **Obvious patterns** - Standard WordPress/PHP conventions
- **Temporary code** - TODOs, experimental features
- **Auto-generated content** - Build artifacts, vendor code
- **Private helper methods** - Internal utilities
- **Exact parameter signatures** - Unless it's a public API contract

## Standard Documentation Structure

Follow this flexible pattern (general → specific → connected):

**1. Title & Overview** (Required)
- Clear title + 2-3 sentence overview

**2. Main Concepts** (Required)
- Key principles/components readers need first

**3. Detailed Description** (Required)
- In-depth content with topic-specific sections:
  - **Architecture:** Components, Data Flow, Lifecycle, Configuration
  - **Plugin:** Bootstrap, MVC, Hooks, Integration
  - **Package:** Build Process, Archive Formats, Tracking
  - **Storage:** Providers, Upload Flow, Chunking
  - **Installer:** Extraction, Database Restore, Search/Replace
  - **Security:** Validation, Sanitization, Capabilities

**4. Related Documentation & References** (Optional)
- Internal links: `[Package System](01_2_ARCH_package-system.md)`
- External links: WordPress Codex, PHP docs, specifications

**Key principle:** Adapt freely - these guide structure, not dictate it. The goal is clarity, not conformity.
